using System;
using System.Data;
using System.Collections;

using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Provides means for getting lab order information from VistA system via VistALink.</summary>

	#endregion
	
	public class LabOrder
	{
		private const string LAB_ORDER_LOOKUP_BY_UID_RPC_NAME = "VBECS LAB ORDER LOOKUP BY UID";
		private const string LAB_ACCESSION_LOOKUP_BY_LAB_ORDER_NUMBER = "VBECS LAB ACCESSION UID LOOKUP";
		private const string LAB_TEST_TABLE_NAME = "LabTest";

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3996"> 
		///		<ExpectedInput>'2932460003' as the Specimen UID.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataSet with expected data including patient
		///		Name, VistaPatientId, AccessionNumber, LabOrderNuber, UID,
		///		and ordered tests.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4000"> 
		///		<ExpectedInput>Null specimenUID parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets VistA lab order number by specimen UID.
		/// </summary>
		/// <param name="specimenUID">Specimen UID.</param>
		/// <returns><see cref="DataSet"/> containing lookup results.</returns>
		public static DataSet GetLabOrderNumberBySpecimenUID( string specimenUID )
		{
			if( specimenUID == null )
				throw( new ArgumentNullException( "specimenUID" ) );

			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetLabOrderBySpecimenUIDRpcRequest( specimenUID ) );

			return DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4777"> 
		///		<ExpectedInput>Valid Lab order number that has been accessioned.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataSet with expected data including patient
		///		Name, VistaPatientId, Ordered Tests, AccessionNumber, LabOrderNuber, UID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4778"> 
		///		<ExpectedInput>NULL as input parameter labOrderNumber</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets lab accession number by lab order number.
		/// </summary>
		/// <param name="labOrderNumber">Non-null lab order number.</param>
		/// <returns><see cref="DataSet"/> containing lookup results.</returns>
		public static DataSet GetLabAccessionByLabOrderNumber( string labOrderNumber )
		{
			if( labOrderNumber == null )
				throw( new ArgumentNullException( "labOrderNumber" ) );

			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetLabAccessionByLabOrderNumberRpcRequest( labOrderNumber ) );

			DataSet ds = DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );
			foreach( DataTable dt in ds.Tables )
			{
				if( dt.TableName == LAB_TEST_TABLE_NAME )
				{
					dt.Columns.Add( new DataColumn( "SpecimenCollectionDate", typeof(DateTime) ) );
					foreach( DataRow dr in dt.Rows )
					{
						if( dr.Table.Columns.Contains("CollectionDateTime") && !dr.IsNull("CollectionDateTime"))
						{
							dr["SpecimenCollectionDate"] = Common.HL7DateFormat.ConvertHL7DateTime( dr["CollectionDateTime"].ToString() );
						}
					}
				}
			}

			return ds;
		}

		/// <summary>
		/// Creates Lab Order Lookup by Specimen UID RpcRequest from a given search value.
		/// </summary>
		/// <param name="searchValue">Specimen UID string</param>
		/// <returns>Lab Order Lookup RpcRequest</returns>
		private static RpcRequest GetLabOrderBySpecimenUIDRpcRequest( string searchValue )
		{
			RpcRequest _labOrderLookupRequest = VistALink.CreateRpcRequest( LAB_ORDER_LOOKUP_BY_UID_RPC_NAME );
			_labOrderLookupRequest.Parameters.Add( new RpcStringParameter( 1, ( searchValue != null ) ? searchValue : String.Empty ) );
			
			return _labOrderLookupRequest;
		}

		/// <summary>
		/// GetLabAccessionByLabOrderNumberRpcRequest
		/// </summary>
		/// <param name="searchValue"></param>
		/// <returns></returns>
		private static RpcRequest GetLabAccessionByLabOrderNumberRpcRequest( string searchValue )
		{
			RpcRequest _labAccessionLookupRequest = VistALink.CreateRpcRequest( LAB_ACCESSION_LOOKUP_BY_LAB_ORDER_NUMBER );
			_labAccessionLookupRequest.Parameters.Add( new RpcStringParameter( 1, ( searchValue != null ) ? searchValue : String.Empty ) );
			
			return _labAccessionLookupRequest;
		}
	}
}